// src/node/utils/ensure-extension-dirs.ts
import { pluralize } from "@directus/utils/node";
import fse from "fs-extra";
import path from "path";
async function ensureExtensionDirs(extensionsPath, types) {
  for (const extensionType of types) {
    const dirPath = path.resolve(extensionsPath, pluralize(extensionType));
    try {
      await fse.ensureDir(dirPath);
    } catch {
      throw new Error(`Extension folder "${dirPath}" couldn't be opened`);
    }
  }
}

// src/shared/constants/extension-types.ts
var APP_EXTENSION_TYPES = ["interface", "display", "layout", "module", "panel", "theme"];
var API_EXTENSION_TYPES = ["hook", "endpoint"];
var HYBRID_EXTENSION_TYPES = ["operation"];
var BUNDLE_EXTENSION_TYPES = ["bundle"];
var EXTENSION_TYPES = [
  ...APP_EXTENSION_TYPES,
  ...API_EXTENSION_TYPES,
  ...HYBRID_EXTENSION_TYPES,
  ...BUNDLE_EXTENSION_TYPES
];
var NESTED_EXTENSION_TYPES = [
  ...APP_EXTENSION_TYPES,
  ...API_EXTENSION_TYPES,
  ...HYBRID_EXTENSION_TYPES
];
var APP_OR_HYBRID_EXTENSION_TYPES = [...APP_EXTENSION_TYPES, ...HYBRID_EXTENSION_TYPES];
var APP_OR_HYBRID_EXTENSION_PACKAGE_TYPES = [
  ...APP_OR_HYBRID_EXTENSION_TYPES,
  ...BUNDLE_EXTENSION_TYPES
];

// src/shared/constants/languages.ts
var EXTENSION_LANGUAGES = ["javascript", "typescript"];

// src/shared/constants/name-regex.ts
var EXTENSION_NAME_REGEX = /^(?:(?:@[^/]+\/)?directus-extension-|@directus\/extension-)(.+)$/;

// src/shared/constants/pkg-key.ts
var EXTENSION_PKG_KEY = "directus:extension";

// src/shared/constants/shared-deps.ts
var APP_SHARED_DEPS = ["@directus/extensions-sdk", "vue", "vue-router", "vue-i18n", "pinia"];
var API_SHARED_DEPS = ["directus", "directus:api"];

// src/node/utils/generate-extensions-entrypoint.ts
import { isIn, isTypeIn, pathToRelativeUrl, pluralize as pluralize2 } from "@directus/utils/node";
import path2 from "path";
function generateExtensionsEntrypoint(extensions) {
  const appOrHybridExtensions = extensions.filter(
    (extension) => isIn(extension.type, [...APP_EXTENSION_TYPES, ...HYBRID_EXTENSION_TYPES])
  );
  const bundleExtensions = extensions.filter(
    (extension) => extension.type === "bundle" && extension.entries.some((entry) => isIn(entry.type, [...APP_EXTENSION_TYPES, ...HYBRID_EXTENSION_TYPES]))
  );
  const appOrHybridExtensionImports = [...APP_EXTENSION_TYPES, ...HYBRID_EXTENSION_TYPES].flatMap(
    (type) => appOrHybridExtensions.filter((extension) => extension.type === type).map(
      (extension, i) => `import ${type}${i} from './${pathToRelativeUrl(
        path2.resolve(
          extension.path,
          isTypeIn(extension, HYBRID_EXTENSION_TYPES) ? extension.entrypoint.app : extension.entrypoint
        )
      )}';`
    )
  );
  const bundleExtensionImports = bundleExtensions.map(
    (extension, i) => `import {${[...APP_EXTENSION_TYPES, ...HYBRID_EXTENSION_TYPES].filter((type) => extension.entries.some((entry) => entry.type === type)).map((type) => `${pluralize2(type)} as ${type}Bundle${i}`).join(",")}} from './${pathToRelativeUrl(path2.resolve(extension.path, extension.entrypoint.app))}';`
  );
  const extensionExports = [...APP_EXTENSION_TYPES, ...HYBRID_EXTENSION_TYPES].map(
    (type) => `export const ${pluralize2(type)} = [${appOrHybridExtensions.filter((extension) => extension.type === type).map((_, i) => `${type}${i}`).concat(
      bundleExtensions.map(
        (extension, i) => extension.entries.some((entry) => entry.type === type) ? `...${type}Bundle${i}` : null
      ).filter((e) => e !== null)
    ).join(",")}];`
  );
  return `${appOrHybridExtensionImports.join("")}${bundleExtensionImports.join("")}${extensionExports.join("")}`;
}

// src/node/utils/get-extensions.ts
import { isIn as isIn2, isTypeIn as isTypeIn2, listFolders, pluralize as pluralize3, resolvePackage } from "@directus/utils/node";
import fse2 from "fs-extra";
import { pick } from "lodash-es";
import path3 from "path";

// src/shared/schemas/manifest.ts
import { z as z2 } from "zod";

// src/shared/schemas/options.ts
import { z } from "zod";
var SplitEntrypoint = z.object({
  app: z.string(),
  api: z.string()
});
var ExtensionSandboxRequestedScopes = z.object({
  request: z.optional(
    z.object({
      urls: z.array(z.string()),
      methods: z.array(
        z.union([z.literal("GET"), z.literal("POST"), z.literal("PATCH"), z.literal("PUT"), z.literal("DELETE")])
      )
    })
  ),
  log: z.optional(z.object({})),
  sleep: z.optional(z.object({}))
});
var ExtensionSandboxOptions = z.optional(
  z.object({
    enabled: z.boolean(),
    requestedScopes: ExtensionSandboxRequestedScopes
  })
);
var ExtensionOptionsBundleEntry = z.union([
  z.object({
    type: z.enum(API_EXTENSION_TYPES),
    name: z.string(),
    source: z.string()
  }),
  z.object({
    type: z.enum(APP_EXTENSION_TYPES),
    name: z.string(),
    source: z.string()
  }),
  z.object({
    type: z.enum(HYBRID_EXTENSION_TYPES),
    name: z.string(),
    source: SplitEntrypoint
  })
]);
var ExtensionOptionsBase = z.object({
  host: z.string(),
  hidden: z.boolean().optional()
});
var ExtensionOptionsApp = z.object({
  type: z.enum(APP_EXTENSION_TYPES),
  path: z.string(),
  source: z.string()
});
var ExtensionOptionsApi = z.object({
  type: z.enum(API_EXTENSION_TYPES),
  path: z.string(),
  source: z.string(),
  sandbox: ExtensionSandboxOptions
});
var ExtensionOptionsHybrid = z.object({
  type: z.enum(HYBRID_EXTENSION_TYPES),
  path: SplitEntrypoint,
  source: SplitEntrypoint,
  sandbox: ExtensionSandboxOptions
});
var ExtensionOptionsBundle = z.object({
  type: z.literal("bundle"),
  path: SplitEntrypoint,
  entries: z.array(ExtensionOptionsBundleEntry)
});
var ExtensionOptionsBundleEntries = z.array(ExtensionOptionsBundleEntry);
var ExtensionOptions = ExtensionOptionsBase.and(
  z.union([ExtensionOptionsApp, ExtensionOptionsApi, ExtensionOptionsHybrid, ExtensionOptionsBundle])
);

// src/shared/schemas/manifest.ts
var ExtensionManifest = z2.object({
  name: z2.string(),
  version: z2.string(),
  type: z2.union([z2.literal("module"), z2.literal("commonjs")]).optional(),
  description: z2.string().optional(),
  icon: z2.string().optional(),
  dependencies: z2.record(z2.string()).optional(),
  [EXTENSION_PKG_KEY]: ExtensionOptions
});

// src/node/utils/get-extensions.ts
var findExtension = async (folder, filename) => {
  if (await fse2.exists(path3.join(folder, `${filename}.cjs`)))
    return `${filename}.cjs`;
  if (await fse2.exists(path3.join(folder, `${filename}.mjs`)))
    return `${filename}.mjs`;
  return `${filename}.js`;
};
async function resolvePackageExtensions(root, extensionNames) {
  const extensions = [];
  const local = extensionNames === void 0;
  if (extensionNames === void 0) {
    extensionNames = await listFolders(root);
    extensionNames = extensionNames.filter((name) => EXTENSION_NAME_REGEX.test(name));
  }
  for (const extensionName of extensionNames) {
    const extensionPath = local ? path3.join(root, extensionName) : resolvePackage(extensionName, root);
    const extensionManifest = await fse2.readJSON(path3.join(extensionPath, "package.json"));
    let parsedManifest;
    try {
      parsedManifest = ExtensionManifest.parse(extensionManifest);
    } catch (error) {
      throw new Error(`The extension manifest of "${extensionName}" is not valid.
${error}`);
    }
    const extensionOptions = parsedManifest[EXTENSION_PKG_KEY];
    if (extensionOptions.type === "bundle") {
      extensions.push({
        path: extensionPath,
        name: parsedManifest.name,
        version: parsedManifest.version,
        type: extensionOptions.type,
        entrypoint: {
          app: extensionOptions.path.app,
          api: extensionOptions.path.api
        },
        entries: extensionOptions.entries.map((entry) => pick(entry, "name", "type")),
        host: extensionOptions.host,
        local
      });
    } else if (isTypeIn2(extensionOptions, HYBRID_EXTENSION_TYPES)) {
      extensions.push({
        path: extensionPath,
        name: parsedManifest.name,
        version: parsedManifest.version,
        type: extensionOptions.type,
        entrypoint: {
          app: extensionOptions.path.app,
          api: extensionOptions.path.api
        },
        host: extensionOptions.host,
        sandbox: extensionOptions.sandbox,
        local
      });
    } else if (extensionOptions.type === "hook" || extensionOptions.type === "endpoint") {
      extensions.push({
        path: extensionPath,
        name: parsedManifest.name,
        version: parsedManifest.version,
        type: extensionOptions.type,
        entrypoint: extensionOptions.path,
        host: extensionOptions.host,
        sandbox: extensionOptions.sandbox,
        local
      });
    } else {
      extensions.push({
        path: extensionPath,
        name: parsedManifest.name,
        version: parsedManifest.version,
        type: extensionOptions.type,
        entrypoint: extensionOptions.path,
        host: extensionOptions.host,
        local
      });
    }
  }
  return extensions;
}
async function getPackageExtensions(root) {
  let pkg;
  try {
    pkg = await fse2.readJSON(path3.resolve(root, "package.json"));
  } catch {
    throw new Error("Current folder does not contain a package.json file");
  }
  const extensionNames = Object.keys(pkg.dependencies ?? {}).filter((dep) => EXTENSION_NAME_REGEX.test(dep));
  return resolvePackageExtensions(root, extensionNames);
}
async function getLocalExtensions(root) {
  const extensions = [];
  for (const extensionType of NESTED_EXTENSION_TYPES) {
    const typeDir = pluralize3(extensionType);
    const typePath = path3.resolve(root, typeDir);
    try {
      const extensionNames = await listFolders(typePath);
      for (const extensionName of extensionNames) {
        const extensionPath = path3.join(typePath, extensionName);
        if (isIn2(extensionType, HYBRID_EXTENSION_TYPES)) {
          extensions.push({
            path: extensionPath,
            name: `${extensionName}:${extensionType}`,
            type: extensionType,
            entrypoint: {
              app: await findExtension(extensionPath, "app"),
              api: await findExtension(extensionPath, "api")
            },
            local: true
          });
        } else {
          extensions.push({
            path: extensionPath,
            name: `${extensionName}:${extensionType}`,
            type: extensionType,
            entrypoint: await findExtension(extensionPath, "index"),
            local: true
          });
        }
      }
    } catch (e) {
      throw new Error(`Extension folder "${typePath}" couldn't be opened`);
    }
  }
  return extensions;
}

// src/shared/utils/define-extension.ts
function defineInterface(config) {
  return config;
}
function defineDisplay(config) {
  return config;
}
function defineLayout(config) {
  return config;
}
function defineModule(config) {
  return config;
}
function definePanel(config) {
  return config;
}
function defineHook(config) {
  return config;
}
function defineEndpoint(config) {
  return config;
}
function defineOperationApp(config) {
  return config;
}
function defineOperationApi(config) {
  return config;
}
export {
  API_EXTENSION_TYPES,
  API_SHARED_DEPS,
  APP_EXTENSION_TYPES,
  APP_OR_HYBRID_EXTENSION_PACKAGE_TYPES,
  APP_OR_HYBRID_EXTENSION_TYPES,
  APP_SHARED_DEPS,
  BUNDLE_EXTENSION_TYPES,
  EXTENSION_LANGUAGES,
  EXTENSION_NAME_REGEX,
  EXTENSION_PKG_KEY,
  EXTENSION_TYPES,
  ExtensionManifest,
  ExtensionOptions,
  ExtensionOptionsApi,
  ExtensionOptionsApp,
  ExtensionOptionsBase,
  ExtensionOptionsBundle,
  ExtensionOptionsBundleEntries,
  ExtensionOptionsBundleEntry,
  ExtensionOptionsHybrid,
  ExtensionSandboxOptions,
  ExtensionSandboxRequestedScopes,
  HYBRID_EXTENSION_TYPES,
  NESTED_EXTENSION_TYPES,
  SplitEntrypoint,
  defineDisplay,
  defineEndpoint,
  defineHook,
  defineInterface,
  defineLayout,
  defineModule,
  defineOperationApi,
  defineOperationApp,
  definePanel,
  ensureExtensionDirs,
  findExtension,
  generateExtensionsEntrypoint,
  getLocalExtensions,
  getPackageExtensions,
  resolvePackageExtensions
};
