"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _collation = require("../collation");

var _token = require("./token");

const types = {
  1: {
    name: 'DATABASE',
    event: 'databaseChange'
  },
  2: {
    name: 'LANGUAGE',
    event: 'languageChange'
  },
  3: {
    name: 'CHARSET',
    event: 'charsetChange'
  },
  4: {
    name: 'PACKET_SIZE',
    event: 'packetSizeChange'
  },
  7: {
    name: 'SQL_COLLATION',
    event: 'sqlCollationChange'
  },
  8: {
    name: 'BEGIN_TXN',
    event: 'beginTransaction'
  },
  9: {
    name: 'COMMIT_TXN',
    event: 'commitTransaction'
  },
  10: {
    name: 'ROLLBACK_TXN',
    event: 'rollbackTransaction'
  },
  13: {
    name: 'DATABASE_MIRRORING_PARTNER',
    event: 'partnerNode'
  },
  17: {
    name: 'TXN_ENDED'
  },
  18: {
    name: 'RESET_CONNECTION',
    event: 'resetConnection'
  },
  20: {
    name: 'ROUTING_CHANGE',
    event: 'routingChange'
  }
};

function readNewAndOldValue(parser, length, type, callback) {
  switch (type.name) {
    case 'DATABASE':
    case 'LANGUAGE':
    case 'CHARSET':
    case 'PACKET_SIZE':
    case 'DATABASE_MIRRORING_PARTNER':
      return parser.readBVarChar(newValue => {
        parser.readBVarChar(oldValue => {
          switch (type.name) {
            case 'PACKET_SIZE':
              return callback(new _token.PacketSizeEnvChangeToken(parseInt(newValue), parseInt(oldValue)));

            case 'DATABASE':
              return callback(new _token.DatabaseEnvChangeToken(newValue, oldValue));

            case 'LANGUAGE':
              return callback(new _token.LanguageEnvChangeToken(newValue, oldValue));

            case 'CHARSET':
              return callback(new _token.CharsetEnvChangeToken(newValue, oldValue));

            case 'DATABASE_MIRRORING_PARTNER':
              return callback(new _token.DatabaseMirroringPartnerEnvChangeToken(newValue, oldValue));
          }
        });
      });

    case 'SQL_COLLATION':
    case 'BEGIN_TXN':
    case 'COMMIT_TXN':
    case 'ROLLBACK_TXN':
    case 'RESET_CONNECTION':
      return parser.readBVarByte(newValue => {
        parser.readBVarByte(oldValue => {
          switch (type.name) {
            case 'SQL_COLLATION':
              {
                const newCollation = newValue.length ? _collation.Collation.fromBuffer(newValue) : undefined;
                const oldCollation = oldValue.length ? _collation.Collation.fromBuffer(oldValue) : undefined;
                return callback(new _token.CollationChangeToken(newCollation, oldCollation));
              }

            case 'BEGIN_TXN':
              return callback(new _token.BeginTransactionEnvChangeToken(newValue, oldValue));

            case 'COMMIT_TXN':
              return callback(new _token.CommitTransactionEnvChangeToken(newValue, oldValue));

            case 'ROLLBACK_TXN':
              return callback(new _token.RollbackTransactionEnvChangeToken(newValue, oldValue));

            case 'RESET_CONNECTION':
              return callback(new _token.ResetConnectionEnvChangeToken(newValue, oldValue));
          }
        });
      });

    case 'ROUTING_CHANGE':
      return parser.readUInt16LE(valueLength => {
        // Routing Change:
        // Byte 1: Protocol (must be 0)
        // Bytes 2-3 (USHORT): Port number
        // Bytes 4-5 (USHORT): Length of server data in unicode (2byte chars)
        // Bytes 6-*: Server name in unicode characters
        parser.readBuffer(valueLength, routePacket => {
          const protocol = routePacket.readUInt8(0);

          if (protocol !== 0) {
            throw new Error('Unknown protocol byte in routing change event');
          }

          const port = routePacket.readUInt16LE(1);
          const serverLen = routePacket.readUInt16LE(3); // 2 bytes per char, starting at offset 5

          const server = routePacket.toString('ucs2', 5, 5 + serverLen * 2);
          const newValue = {
            protocol: protocol,
            port: port,
            server: server
          };
          parser.readUInt16LE(oldValueLength => {
            parser.readBuffer(oldValueLength, oldValue => {
              callback(new _token.RoutingEnvChangeToken(newValue, oldValue));
            });
          });
        });
      });

    default:
      console.error('Tedious > Unsupported ENVCHANGE type ' + type.name); // skip unknown bytes

      parser.readBuffer(length - 1, () => {
        callback(undefined);
      });
  }
}

function envChangeParser(parser, _options, callback) {
  parser.readUInt16LE(length => {
    parser.readUInt8(typeNumber => {
      const type = types[typeNumber];

      if (!type) {
        console.error('Tedious > Unsupported ENVCHANGE type ' + typeNumber); // skip unknown bytes

        return parser.readBuffer(length - 1, () => {
          callback(undefined);
        });
      }

      readNewAndOldValue(parser, length, type, token => {
        callback(token);
      });
    });
  });
}

var _default = envChangeParser;
exports.default = _default;
module.exports = envChangeParser;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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